/**
 * Copyright (c) 2016 CA, All rights Reserved.

This software and all information contained therein is confidential and 
proprietary and shall not be duplicated, used, disclosed or disseminated in any 
way except as authorized by the applicable license agreement, without the 
express written permission of CA. All authorized reproductions must be marked 
with this language.  

EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT PERMITTED 
BY APPLICABLE LAW, CA PROVIDES THIS SOFTWARE WITHOUT WARRANTY OF ANY KIND, 
INCLUDING WITHOUT LIMITATION, ANY IMPLIED WARRANTIES OF MERCHANTABILITY OR 
FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT WILL CA BE LIABLE TO THE END USER 
OR ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE OF 
THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, LOST PROFITS, BUSINESS 
INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF CA IS EXPRESSLY ADVISED OF SUCH 
LOSS OR DAMAGE.

 ***********************************************************************/
package com.ca.fmp.ims.view.ckey;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.TreeSet;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.StyledCellLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.TextLayout;
import org.eclipse.swt.graphics.TextStyle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;
import org.osgi.framework.Bundle;

import com.ca.fmp.ims.common.Constants;
import com.ca.fmp.ims.model.generated.CKeyEntryType;
import com.ca.fmp.ims.model.generated.CKeyFieldType;
import com.ca.fmp.ims.model.generated.DisplayModeType;
import com.ca.fmp.ims.model.generated.YorNType;
import com.ca.fmp.ims.view.editor.EditorView;

public class ConcatenatedKeyView extends ViewPart  {

	public static String ID = "com.ca.fmp.ims.view.ckey";
	private static String CONTEXT_HELP_ID = "com.ca.fmp.ims.help.ConcatenatedKeyView";
	
	private static final String HEX_BUTTON_ID = "com.ca.fmp.ims.view.commands.ckey.hex";
	private static final String CHAR_BUTTON_ID = "com.ca.fmp.ims.view.commands.ckey.charFormat";
	private static final String SF_BUTTON_ID = "com.ca.fmp.ims.view.commands.ckey.sfFormat";
	
	private Action hexAction;
	private Action charFormatAction;
	private Action sfFormatAction;
	
	private IToolBarManager toolbarManager;
	private IMenuManager menuManager;
	private MenuManager popupMenuMgr;
	
	private YorNType ckeyHex = YorNType.N;
	private DisplayModeType ckeyDisplay = DisplayModeType.C;

	private TableViewer viewer;
	private Composite parentComposite;
	
	private String segid;

	public void createPartControl(Composite parent) {
		parentComposite = parent;
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parentComposite, CONTEXT_HELP_ID);
		
		Bundle bundle = Platform.getBundle("com.ca.fmp.ims.view");
		ImageDescriptor charImage = ImageDescriptor.createFromURL(FileLocator.find(bundle, new Path("icons/editor-open-character-16.png"), null));
		ImageDescriptor singleImage = ImageDescriptor.createFromURL(FileLocator.find(bundle, new Path("icons/editor-open-single-16.png"), null));
		ImageDescriptor hexImage = ImageDescriptor.createFromURL(FileLocator.find(bundle, new Path("icons/hex-mode-16.png"), null));
		
		hexAction = new Action("Hexadecimal") {
			@Override
			public void run() {
				IViewReference[] viewReferences = getSite().getWorkbenchWindow().getActivePage().getViewReferences();
				for(int i = 0; i < viewReferences.length; i++){
					if(viewReferences[i].getId().startsWith("com.ca.fmp.ims.view.editor.EditorView")){
						
						if(ckeyHex == YorNType.Y){
							ckeyHex = YorNType.N;
							if(ckeyDisplay == DisplayModeType.C){
								viewer.getTable().dispose();
								createCharTable();
							}
							((EditorView)viewReferences[i].getView(false)).getFormatMode().loadCurrentSegmentsWithCKEYInfo(ckeyDisplay, ckeyHex);
							hexAction.setChecked(false);
						} else {
							ckeyHex = YorNType.Y;
							((EditorView)viewReferences[i].getView(false)).getFormatMode().loadCurrentSegmentsWithCKEYInfo(ckeyDisplay, ckeyHex);
							hexAction.setChecked(true);
						}
						
						break;
					}
				}
			}
		};	
		hexAction.setId(HEX_BUTTON_ID);
		hexAction.setActionDefinitionId(HEX_BUTTON_ID);
		getViewSite().getActionBars().setGlobalActionHandler(HEX_BUTTON_ID, hexAction);
		hexAction.setImageDescriptor(hexImage);
		hexAction.setToolTipText("Hexadecimal");
		hexAction.setChecked(false);

		charFormatAction = new Action("Character Format") {
			@Override
			public void run() {
				IViewReference[] viewReferences = getSite().getWorkbenchWindow().getActivePage().getViewReferences();
				for(int i = 0; i < viewReferences.length; i++){
					if(viewReferences[i].getId().startsWith("com.ca.fmp.ims.view.editor.EditorView")){
						viewer.getTable().dispose();
						createCharTable();
						
						ckeyDisplay = DisplayModeType.C;
						((EditorView)viewReferences[i].getView(false)).getFormatMode().loadCurrentSegmentsWithCKEYInfo(ckeyDisplay, ckeyHex);
						
						charFormatAction.setEnabled(false);
						sfFormatAction.setEnabled(true);
						break;
					}
				}
			}
		};
		charFormatAction.setImageDescriptor(charImage);
		charFormatAction.setId(CHAR_BUTTON_ID);
		charFormatAction.setActionDefinitionId(CHAR_BUTTON_ID);
		getViewSite().getActionBars().setGlobalActionHandler(CHAR_BUTTON_ID, charFormatAction);	
		
		sfFormatAction = new Action("Single Record Format") {
			@Override
			public void run() {
				IViewReference[] viewReferences = getSite().getWorkbenchWindow().getActivePage().getViewReferences();
				for(int i = 0; i < viewReferences.length; i++){
					if(viewReferences[i].getId().startsWith("com.ca.fmp.ims.view.editor.EditorView")){
						viewer.getTable().dispose();
						createSFTable();
						
						ckeyDisplay = DisplayModeType.S;
						((EditorView)viewReferences[i].getView(false)).getFormatMode().loadCurrentSegmentsWithCKEYInfo(ckeyDisplay, ckeyHex);
						
						charFormatAction.setEnabled(true);
						sfFormatAction.setEnabled(false);
						break;
					}
				}	
			}
		};
		sfFormatAction.setImageDescriptor(singleImage);
		sfFormatAction.setId(SF_BUTTON_ID);
		sfFormatAction.setActionDefinitionId(SF_BUTTON_ID);
		getViewSite().getActionBars().setGlobalActionHandler(SF_BUTTON_ID, sfFormatAction);	
				
		menuManager = getViewSite().getActionBars().getMenuManager();
		menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new IMenuListener() {
			@Override
			public void menuAboutToShow(IMenuManager m) {
				menuManager.add(charFormatAction);
				menuManager.add(sfFormatAction);
				menuManager.add(hexAction);
			}
	    });
		
		popupMenuMgr = new MenuManager("#PopupMenu");
		popupMenuMgr.setRemoveAllWhenShown(true);
		popupMenuMgr.addMenuListener(new IMenuListener() {
			@Override
			public void menuAboutToShow(IMenuManager m) {
				popupMenuMgr.add(charFormatAction);
				popupMenuMgr.add(sfFormatAction);	
				popupMenuMgr.add(hexAction);
			}					
	    });
		
		toolbarManager = getViewSite().getActionBars().getToolBarManager();
		toolbarManager.add(charFormatAction);
		toolbarManager.add(sfFormatAction);
		toolbarManager.add(hexAction);
		
		boolean editorExists = false;
		
		IViewReference[] viewReferences = getSite().getWorkbenchWindow().getActivePage().getViewReferences();
		
		for(int i = 0; i < viewReferences.length; i++){
			if(viewReferences[i].getId().startsWith("com.ca.fmp.ims.view.editor.EditorView")){
				EditorView editorView = (EditorView)viewReferences[i].getView(false);
				if (editorView.getFormatMode() == null){
					break;
				}
				ckeyDisplay = editorView.getDisplayMode();
				ckeyHex = editorView.getHexToggled();
				
				if(ckeyDisplay == DisplayModeType.C){
					createCharTable();
				} else if(ckeyDisplay == DisplayModeType.S){
					createSFTable();
				}
				
				if(ckeyHex == YorNType.Y){
					hexAction.setChecked(true);
				} else {
					hexAction.setChecked(false);
				}
				
				editorView.getFormatMode().loadCurrentSegmentsWithCKEYInfo(ckeyDisplay, ckeyHex);
				
				editorExists = true;
				break;
			}
		}
		
		if(editorExists == false){
			createCharTable();
		}
	}
			
	@Override
	public void setFocus() {}
	
	public YorNType getCKeyHex(){
		return ckeyHex;
	}

	public void updateCKey(List<CKeyEntryType> cKeyList){
		viewer.getTable().removeAll();
		viewer.getTable().setRedraw(false);
		int max_size = 0;
		
		if(ckeyDisplay == DisplayModeType.C){
			if(cKeyList != null){
				TableItem item;
				String charData;

				if(ckeyHex == YorNType.N){
					for(CKeyEntryType cKeyEntry : cKeyList){
						item = new TableItem (viewer.getTable(), SWT.NONE);

						item.setText(0, cKeyEntry.getSegname());

						charData = "";
						for(CKeyFieldType cKeyField : cKeyEntry.getCKeyField()){
							charData = charData.concat(cKeyField.getCharData());
						}
						item.setText(1, charData);
						item.setData(cKeyEntry);
					}
				} else {
					String hexData1, hexData2;
					max_size = viewer.getTable().getColumns()[1].getText().length();
					for(CKeyEntryType cKeyEntry : cKeyList){
						item = new TableItem (viewer.getTable(), SWT.NONE);

						item.setText(0, cKeyEntry.getSegname());

						charData = "";
						hexData1 = "";
						hexData2 = "";
						for(CKeyFieldType cKeyField : cKeyEntry.getCKeyField()){
							charData = charData.concat(cKeyField.getCharData());
							hexData1 = hexData1.concat(cKeyField.getHexData1());
							hexData2 = hexData2.concat(cKeyField.getHexData2());
							if (max_size <= cKeyField.getCharData().length()) { 
								max_size = cKeyField.getCharData().length();
							}
						}
						item.setText(1, charData + Constants.LINE_SEPARATOR + hexData1 + Constants.LINE_SEPARATOR + hexData2);
						item.setData(cKeyEntry);
					}
				}
			}

			if (max_size != 0) {
				viewer.getTable().getColumn(1).setWidth((max_size+1+1)*Constants.getPixelSizeOfCharacter());
			} else {
			 viewer.getTable().getColumn(1).pack();
			}
		}
		else if(ckeyDisplay == DisplayModeType.S){
			if(cKeyList != null){
				TableItem item;

				for(CKeyEntryType cKeyEntry : cKeyList){
					String segmentName = cKeyEntry.getSegname();
					
					int count = cKeyEntry.getCKeyField().size();
					
					//Field Name
					TreeSet<String> fieldLevelSet = new TreeSet<String>();
					String fieldLevelRepresentedAstwoDigitString;
					for (int i = 0; i < count; i++){	
						fieldLevelRepresentedAstwoDigitString = (cKeyEntry.getCKeyField().get(i).getLevel() < 10 ? "0" : "")
								+ String.valueOf(cKeyEntry.getCKeyField().get(i).getLevel()); 
						fieldLevelSet.add(fieldLevelRepresentedAstwoDigitString);
					}
					ArrayList<String> fieldLevels = new ArrayList<String>(fieldLevelSet);
					ArrayList<String> indentedFieldNames = new ArrayList<String>();
					for (int i = 0; i < count; i++){	
						fieldLevelRepresentedAstwoDigitString = (cKeyEntry.getCKeyField().get(i).getLevel() < 10 ? "0" : "")
								+ String.valueOf(cKeyEntry.getCKeyField().get(i).getLevel());			
						int indentation = fieldLevels.indexOf(fieldLevelRepresentedAstwoDigitString);
						char[] array = new char[indentation * 2];
						Arrays.fill(array, ' ');
						indentedFieldNames.add(new String(array)
							+ fieldLevelRepresentedAstwoDigitString + " "
							+ String.valueOf(cKeyEntry.getCKeyField().get(i).getName()));
					}
					
					//Format 
					ArrayList<String> formatStrings = new ArrayList<String>();
					for (int i = 0; i < count; i++){
						CKeyFieldType cKeyField = cKeyEntry.getCKeyField().get(i);
						String fieldFormatForDisplay = cKeyField.getFormatChar();
						if (fieldFormatForDisplay == null) {
							fieldFormatForDisplay = "";
						}
						
						// floating point type, response form the mainframe is 1 or 2
						if (fieldFormatForDisplay == "1" || fieldFormatForDisplay == "2") {
							fieldFormatForDisplay = "FP";
						}
						
						// unsigned or signed format type; if signed add 'S' to format 
						if (cKeyField.getSigned() == YorNType.Y) {
							fieldFormatForDisplay += "S";
						}
						
						// length of some format types can consist of two parts, integer and fraction
						String fieldFormatMaxLengthDisplay = "";

						if (cKeyField.getIntegerLength() == null) {
							fieldFormatMaxLengthDisplay = cKeyField.getLength().toString();
						} else {
							fieldFormatMaxLengthDisplay = cKeyField.getIntegerLength().toString();
							if (cKeyField.getFractionLength() != null) {
								fieldFormatMaxLengthDisplay += "." + cKeyField.getFractionLength().toString();
							}
						}
						
						// Format column consists at last 7 characters 
						int charFormatLength = fieldFormatForDisplay.length() + fieldFormatMaxLengthDisplay.length();			
						if (charFormatLength <= 7) {
							char[] array = new char[7 - charFormatLength];
							Arrays.fill(array, ' ');
							formatStrings.add(fieldFormatForDisplay + new String(array) + fieldFormatMaxLengthDisplay);
						} else {
							formatStrings.add(fieldFormatForDisplay + " " + fieldFormatMaxLengthDisplay);
						}
					}
					
					for(int i = 0; i < cKeyEntry.getCKeyField().size(); i++){
						CKeyFieldType cKeyField = cKeyEntry.getCKeyField().get(i);
						item = new TableItem (viewer.getTable(), SWT.NONE);

						if(i == 0){
							item.setText(0, segmentName);
						} else {
							item.setText(0, "");
						}
						
						item.setText(1, indentedFieldNames.get(i));
						item.setText(2, String.valueOf(cKeyField.getPosition()));
						item.setText(3, formatStrings.get(i));
						item.setText(4, (cKeyField.getCharData() == null) ? "" : cKeyField.getCharData());
						item.setData(cKeyField);
					}
				}
			}

			viewer.getTable().getColumn(0).pack();
			viewer.getTable().getColumn(1).pack();
			viewer.getTable().getColumn(2).pack();
			viewer.getTable().getColumn(3).pack();
			viewer.getTable().getColumn(4).pack();
		}
		
		viewer.getTable().setRedraw(true);
	}


	private void updateControlsState(){
		boolean isEditorOpened = false;
		
		IViewReference[] viewReferences = getSite().getWorkbenchWindow().getActivePage().getViewReferences();
		for(int i = 0; i < viewReferences.length; i++){
			if(viewReferences[i].getId().startsWith("com.ca.fmp.ims.view.editor.EditorView")){
				isEditorOpened = true;
				break;
			}
		}
		
		if(isEditorOpened == false){
			hexAction.setChecked(false);
			hexAction.setEnabled(false);
			charFormatAction.setEnabled(false);
			sfFormatAction.setEnabled(false);
		} else {
			if(ckeyDisplay == DisplayModeType.C){
				charFormatAction.setEnabled(false);
				sfFormatAction.setEnabled(true);
			} else if(ckeyDisplay == DisplayModeType.S){
				charFormatAction.setEnabled(true);
				sfFormatAction.setEnabled(false);
			}
		}
	}
	
	public void wakeUpView(DisplayModeType displayMode){
		ckeyDisplay = displayMode;
		hexAction.setEnabled(true);
		if(ckeyDisplay == DisplayModeType.C){
			charFormatAction.setEnabled(false);
			sfFormatAction.setEnabled(true);
			
			if(viewer.getTable().getColumnCount() > 2){
				viewer.getTable().dispose();
				createCharTable();
			}
		} else if(ckeyDisplay == DisplayModeType.S){
			charFormatAction.setEnabled(true);
			sfFormatAction.setEnabled(false);
			
			if(viewer.getTable().getColumnCount() == 2){
				viewer.getTable().dispose();
				createSFTable();
			}
		}
	}
	
	public void disableView(){
		hexAction.setChecked(false);
		hexAction.setEnabled(false);
		charFormatAction.setEnabled(false);
		sfFormatAction.setEnabled(false);
		
		if(ckeyDisplay == DisplayModeType.C && ckeyHex == YorNType.Y){
			ckeyHex = YorNType.N;
			viewer.getTable().dispose();
			createCharTable();
		}
		
		segid = null;
	}
	
	private void createCharTable(){
		viewer = new TableViewer(parentComposite, SWT.BORDER | SWT.MULTI | SWT.FULL_SELECTION);

		final Table table = viewer.getTable();
		table.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		Shell shell = new Shell();
		table.setFont(JFaceResources.getTextFont());

		final TextLayout segmentLayout = new TextLayout(shell.getDisplay());
		TableViewerColumn colSegment = new TableViewerColumn(viewer, SWT.NONE);
		colSegment.getColumn().setText("Segment  ");
		colSegment.getColumn().setResizable(true);
		colSegment.getColumn().pack();
		colSegment.setLabelProvider(new StyledCellLabelProvider(){
			@Override
			protected void paint(Event event, Object element) {
				segmentLayout.setFont(JFaceResources.getTextFont());
				segmentLayout.setText(((TableItem)event.item).getText(0));
				segmentLayout.draw(event.gc, event.x, event.y);
			}
		});

		TextLayout keyValueLayout = new TextLayout(shell.getDisplay());
		TableViewerColumn colKeyValue = new TableViewerColumn(viewer, SWT.NONE);
		colKeyValue.getColumn().setText("Key Value");
		colKeyValue.getColumn().setResizable(true);
		colKeyValue.getColumn().pack();
		colKeyValue.setLabelProvider(new ConcatenatedKeyValueLabelProvider(keyValueLayout, shell));
		
		Menu charPopupMenu = popupMenuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(charPopupMenu);
		getSite().registerContextMenu(popupMenuMgr, viewer);
		
		parentComposite.layout();
		
		updateControlsState();
	}
	
	private void createSFTable(){
		viewer = new TableViewer(parentComposite, SWT.BORDER | SWT.FULL_SELECTION);

		final Table table = viewer.getTable();
		final Font font = JFaceResources.getTextFont();
		table.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		table.setFont(JFaceResources.getTextFont());
		viewer.getTable().setFont(JFaceResources.getTextFont());
		
		String[] columnNames = new String[] { "Segment", "Field Name", "Position", "Format", "Key Value" };
		int[] columnAlignments = new int[] { SWT.LEFT, SWT.LEFT, SWT.RIGHT, SWT.CENTER, SWT.LEFT };

		for (int i = 0; i < columnNames.length; i++) {
			TableViewerColumn tableViewerColumn = new TableViewerColumn(viewer, columnAlignments[i]);
			TableColumn tableColumn = tableViewerColumn.getColumn();

			tableColumn.setText(columnNames[i]);
			tableColumn.pack();
			
			if(i == 4){
				final Shell shell = new Shell();
				final TextLayout ckeyValueLayout = new TextLayout(shell.getDisplay());
				tableViewerColumn.setLabelProvider(new StyledCellLabelProvider(){
					@Override
					protected void paint(Event event, Object element) {
						String keyValue = ((TableItem)event.item).getText(4);
						ckeyValueLayout.setFont(font);
						ckeyValueLayout.setText(keyValue);
						
						if(ckeyHex == YorNType.N
								&& ((TableItem)event.item).getData() instanceof CKeyFieldType
								&& (((CKeyFieldType)((TableItem)event.item).getData()).getLength() * 2) + 3 == keyValue.length()){

							TextStyle colorData = new TextStyle(font, shell.getDisplay().getSystemColor(SWT.COLOR_RED), null);
							ckeyValueLayout.setStyle(colorData, 0, keyValue.length() - 1);
						}
						
						ckeyValueLayout.draw(event.gc, event.x + 2, event.y + 2);
					}
				});
			}
		}
		
		Menu charPopupMenu = popupMenuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(charPopupMenu);
		getSite().registerContextMenu(popupMenuMgr, viewer);
		
		parentComposite.layout();
		
		updateControlsState();
	}
	
	public boolean isEmpty(){
		if(viewer.getTable().getItemCount() == 0){
			return true;
		} else {
			return false;
		}
	}
	
	public void setSegid(String segid){
		this.segid = segid;
	}
	
	public String getSegid(){
		return segid;
	}
}
