/**
 * Copyright (c) 2016 CA, All rights Reserved.

This software and all information contained therein is confidential and 
proprietary and shall not be duplicated, used, disclosed or disseminated in any 
way except as authorized by the applicable license agreement, without the 
express written permission of CA. All authorized reproductions must be marked 
with this language.  

EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT PERMITTED 
BY APPLICABLE LAW, CA PROVIDES THIS SOFTWARE WITHOUT WARRANTY OF ANY KIND, 
INCLUDING WITHOUT LIMITATION, ANY IMPLIED WARRANTIES OF MERCHANTABILITY OR 
FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT WILL CA BE LIABLE TO THE END USER 
OR ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE OF 
THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, LOST PROFITS, BUSINESS 
INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF CA IS EXPRESSLY ADVISED OF SUCH 
LOSS OR DAMAGE.

 ***********************************************************************/
package com.ca.fmp.ims.view.handlers;

import java.util.ArrayList;

import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Text;

public class LocateKeyKeyListener implements KeyListener{
	
	public static final String LINE_SEPARATOR = "\r\n";
	public static final String EMPTY_SPACE_IN_CHAR = " ";
	public static final int EMPTY_SPACE_TO_INSERT = 0x20;

	@SuppressWarnings("unused")
	private TextCellEditor textCellEditorFileMaster;
	private ArrayList<Integer> protectedValues;
	
	private static  boolean insertToggleFlag = false; // determines if we are in insert mode
	
	private long maxLengthP;
	
	private String textText = null;
	private Point textSelection;
	private int textCaretPosition;
	private int textCaretPositionRowIndex; // which row is the caret position in.  In CharMode, there are three rows.
	private int textCaretPositionOffset; // the cursor position based on the row.
	private String[] textFromWidgetSplitIntoRows;
	
	private Text textFromKeyPressed;

	private boolean isDirty = false;
	
	public LocateKeyKeyListener(TextCellEditor textCellEditor) {
		this.textCellEditorFileMaster = textCellEditor;
	}
	
	@Override
	public void keyPressed(KeyEvent keyEvent) {
		// TODO Auto-generated method stub
		
		
		if(keyEvent.doit == false) {
			return;
		}
		
		// Handler for INSERT
		// Set the insertToggleFlag flag
		if(keyEvent.keyCode == SWT.INSERT) {
			insertToggleFlag = !insertToggleFlag;
			keyEvent.doit = false;
			return;
		}
		
		// Initialize variables 
		initializeVariables((Text) keyEvent.getSource());
		
		// Handler for PAGE DOWN 
		if(keyEvent.keyCode == SWT.PAGE_DOWN) {
		/*	textCellEditorFileMaster.deactivate();
			keyEvent.doit = false;
			return;*/
		}		
		
		// Handler for PAGE UP
		if(keyEvent.keyCode == SWT.PAGE_UP) {
		/*	textCellEditorFileMaster.deactivate();
			keyEvent.doit = false;
			return;*/
		}
		
		
		if((keyEvent.stateMask & SWT.CTRL) == SWT.CTRL) {
			keyEvent.doit = false;
            return;
		}
		
		// CTRL-Z (UNDO)
		if (Character.toUpperCase((char)keyEvent.keyCode) == 'Z' && (keyEvent.stateMask & SWT.CTRL) == SWT.CTRL){ 
			keyEvent.doit = false;
		}
				
		// Handler for ARROW LEFT
		if(keyEvent.keyCode == SWT.ARROW_LEFT && keyEvent.stateMask == 0) {
			if(textCaretPositionOffset == 0) {
				keyEvent.doit = false;
				return;
			}
			if(textCaretPosition == 0) {
				keyEvent.doit = false;
				return;
			}
			moveToPreviousCharacter((Text) keyEvent.getSource(), ((Text) keyEvent.getSource()).getCaretPosition(), protectedValues);
			keyEvent.doit = false;
			return;
		}
		
		// Handler for ARROW RIGHT
		if(keyEvent.keyCode == SWT.ARROW_RIGHT && keyEvent.stateMask == 0) {	
			isEndOfRecordField(keyEvent);
			if(keyEvent.doit == false) {
				return;
			}
			moveToNextCharacter((Text) keyEvent.getSource(), ((Text) keyEvent.getSource()).getCaretPosition(), protectedValues);
			keyEvent.doit = false;
			return;
		}
		
		// Handler for BACKSPACE
		if(keyEvent.keyCode == SWT.BS ) {
			int selectionCount = ((Text) keyEvent.getSource()).getSelectionCount();

			if(textCaretPositionOffset == 0 && selectionCount == 0) {
				keyEvent.doit = false;
				return;
			} 

            //If text is selected 		
			if(selectionCount > 0) {
				// If text is protected do nothing
				if(protectedValues.contains(textCaretPosition)) {
					keyEvent.doit = false;
					return;
				}
				handleDeleteKeyOnSelection((Text) keyEvent.getSource(), protectedValues);
			} else {
				// If text is protected do nothing
				if(protectedValues.contains(textCaretPosition - 1)) {
					keyEvent.doit = false;
					return;
				}
				handleDeleteKey(textFromKeyPressed, textCaretPositionOffset-1, protectedValues);
			}
			keyEvent.doit = false;
			return;
		}
		
		
		// Handler for DELETE
		if(keyEvent.keyCode == SWT.DEL) {			
			int selectionCount = textFromKeyPressed.getSelectionCount();
			
			//If text is selected
			if(selectionCount > 0) {
				handleDeleteKeyOnSelection(textFromKeyPressed, protectedValues);
			} else {	
				if(protectedValues.contains(textCaretPosition)) {
					keyEvent.doit = false;
					return;
				} 
				
				Character characterToBeReplaced = getCharacterFromTextControl(textText);
				if(characterToBeReplaced != null && (characterToBeReplaced == '\r' || characterToBeReplaced == '\n')) {
					keyEvent.doit = false;
					return;
				}
				if(characterToBeReplaced == null) {
//					keyEvent.doit = false;
					return;
				}
				
				isEndOfRecordField(keyEvent);
				if(keyEvent.doit == false) {
					return;
				}
				// delete the next key after the cursor
				// we need to stay at the same caretPosition
				handleDeleteKey(textFromKeyPressed, textCaretPositionOffset, protectedValues);		
			}
			keyEvent.doit = false;
			return;
		}
		
		if(keyEvent.character != '\0' && keyEvent.doit == true) {
			if(textFromKeyPressed.getSelectionCount() > 0) {
				handleDeleteKeyOnSelection(textFromKeyPressed, protectedValues);
				if(textCaretPosition == textSelection.y) {
					textCaretPosition = textCaretPosition - Math.abs(textSelection.x - textSelection.y);
				}
			}

			Character characterToBeReplaced = getCharacterFromTextControl(textText);
			if(characterToBeReplaced != null && (characterToBeReplaced == '\r' || characterToBeReplaced == '\n') && insertToggleFlag == false) {
				keyEvent.doit = false;
				return;			
			}	
			if(characterToBeReplaced == null) {
				return;
			}

			// If protected do not overwrite
			if(protectedValues.contains(textCaretPosition)) {
				keyEvent.doit = false;
				return;
			} 
			
			handleNonNullCharacter(keyEvent);
			return;
		}
	}

	@Override
	public void keyReleased(KeyEvent keyEvent) {
		// TODO Auto-generated method stub
		
	}
	
	public void initializeVariables(Text text) {
		textFromKeyPressed = text;
		textText = textFromKeyPressed.getText();
		textCaretPosition = textFromKeyPressed.getCaretPosition();
		textSelection = textFromKeyPressed.getSelection();
		textCaretPositionRowIndex = getRowIndexBasedOnText(text, text.getCaretPosition());
		textCaretPositionOffset = getRowPositionBasedOnText(text, text.getCaretPosition());
		textFromWidgetSplitIntoRows = textFromKeyPressed.getText().split(LINE_SEPARATOR);
	}
	
	private int getRowIndexBasedOnText(Text text, int cursorPosition) {
		
		// trying to determine which row the cursor position is in.
		String[] row = text.getText().split(LINE_SEPARATOR);
		if(row.length == 1) {
			return 0;
		}
		
		// there are more then one row.  hexData is included
		int charDataLength = row[0].length() + LINE_SEPARATOR.length();
		int hexData1Length = charDataLength + row[1].length() + LINE_SEPARATOR.length();
		if(cursorPosition < charDataLength) {
			return 0;
		} else if(cursorPosition < hexData1Length) {
			return 1;
		} else {
			return 2;
		}
	}
	
	protected int getRowPositionBasedOnText(Text text, int cursorPosition) {
		String[] row = text.getText().split(LINE_SEPARATOR);
		
		if(row.length == 1) {
			return cursorPosition;
		}
		
		int charDataLength = row[0].length() + LINE_SEPARATOR.length();
		int hexData1Length = charDataLength + row[1].length() + LINE_SEPARATOR.length();
		
		if(cursorPosition < charDataLength) {
			return cursorPosition;
		} else if(cursorPosition < hexData1Length) {
			return cursorPosition - charDataLength;
		} else {
			return cursorPosition - hexData1Length;
		}
	}
	
	/**
	 * move to the next position
	 * should handle the protected list
	 * 
	 * @param text
	 * @param currentCaretPosition
	 */
	protected int moveToNextCharacter(Text text, int currentCaretPosition, ArrayList<Integer> protectedValues) {
		boolean caretOnProtected = false;
		int nextCaretPosition = currentCaretPosition;
		
		while(protectedValues.contains(new Integer(nextCaretPosition))) {
			nextCaretPosition++;
			caretOnProtected = true;
		}
		
		if (!caretOnProtected){
			nextCaretPosition++;
		}

		text.setSelection(nextCaretPosition, nextCaretPosition);
		return nextCaretPosition;
	}
	
	
	private int moveToPreviousCharacter(Text text, int currentCaretPosition, ArrayList<Integer> protectedValues) {
		boolean caretOnProtected = false;
		if(currentCaretPosition == 0) {
			return currentCaretPosition;
		}
		int previousCaretPosition = currentCaretPosition;
		
		if(protectedValues == null) {
			previousCaretPosition--;
			text.setSelection(previousCaretPosition, previousCaretPosition);
			return previousCaretPosition;
		}
		
		// more to the previous position if protected data
		while(protectedValues.contains(new Integer(previousCaretPosition - 1))) {
			previousCaretPosition--;
			caretOnProtected = true;
		} 
		
		if (!caretOnProtected){
			previousCaretPosition--;
		}

		text.setSelection(previousCaretPosition, previousCaretPosition);
		return previousCaretPosition;		
	}
	
	
	private synchronized void handleDeleteKeyOnSelection(Text text, ArrayList<Integer> characterPositionsThatCantBeModified) {
		int numberOfCharactersToDelete = 0;
		int selectedFromLeftToRightNumOfChar = 0;
		if(textCaretPosition == textSelection.x) {
			// characters selected from right to left - delete the characters following currentCaretPosition
			numberOfCharactersToDelete = textSelection.y - textCaretPosition;
		} else if(textCaretPosition == textSelection.y) {
			// characters selected from left to right - delete the characters previous to currentCaretPosition
			numberOfCharactersToDelete = textCaretPosition - textSelection.x;			
			selectedFromLeftToRightNumOfChar = numberOfCharactersToDelete;
			
		} else {
			return;
		}
		
		if (textFromWidgetSplitIntoRows[0].length() < (textSelection.y - textSelection.x)) {
			return;
		}
		
		// need to check if any of the characters is protected, if so, don't do a delete.
		for(int x=0; x < numberOfCharactersToDelete; x++) {
			if(characterPositionsThatCantBeModified.contains(new Integer(textCaretPosition + x - selectedFromLeftToRightNumOfChar))) {				
				return;
			}
		}
		
		int offset = textCaretPositionRowIndex * (textFromWidgetSplitIntoRows[textCaretPositionRowIndex] + LINE_SEPARATOR).length();
		
		for(int x=0; x < numberOfCharactersToDelete; x++) {
			handleDeleteKey(text, textCaretPosition - selectedFromLeftToRightNumOfChar - offset, protectedValues);
		}
	}
	
	/**
	 * The protected range is not being protected
	 * a record can have three rows (chardata/hex1/hex2)
	 * determine which row you are editing
	 * this is what should be passed to the isInsertable method.		
	 * 
	 * @param currentPositionButNotRowAware current position based ont the Text widget
	 * @param currentPositionButNotRowAware should be the offset value
	 */
	private void handleDeleteKey(Text text, int currentPositionButNotRowAware, ArrayList<Integer> characterPositionsThatCantBeModified) {

		String firstParameter = textFromWidgetSplitIntoRows[textCaretPositionRowIndex].substring(0, currentPositionButNotRowAware);
		String secondParameter = textFromWidgetSplitIntoRows[textCaretPositionRowIndex].substring(currentPositionButNotRowAware + 1, textFromWidgetSplitIntoRows[textCaretPositionRowIndex].length());
		textFromWidgetSplitIntoRows[textCaretPositionRowIndex] = firstParameter + secondParameter;

        int terminatingCharacterPosition = getLastCursorPositionForDeletion(textFromWidgetSplitIntoRows[textCaretPositionRowIndex], textCaretPosition, protectedValues);
        int terminatingCharacterPositionOffset = terminatingCharacterPosition;
       
        String before = textFromWidgetSplitIntoRows[textCaretPositionRowIndex].substring(0, terminatingCharacterPositionOffset -1);
        String after = textFromWidgetSplitIntoRows[textCaretPositionRowIndex].substring(terminatingCharacterPositionOffset - 1, textFromWidgetSplitIntoRows[textCaretPositionRowIndex].length());
		textFromWidgetSplitIntoRows[textCaretPositionRowIndex] = before + EMPTY_SPACE_IN_CHAR + after;        	
        	
		if(textFromWidgetSplitIntoRows.length == 1) {
			text.setText(textFromWidgetSplitIntoRows[0]);
		} else {
			text.setText(textFromWidgetSplitIntoRows[0] + LINE_SEPARATOR + textFromWidgetSplitIntoRows[1] + LINE_SEPARATOR + textFromWidgetSplitIntoRows[2]);
		}
		int currentPosition = getTruePositionBasedOnRowIndexAndPositionOffset(textCaretPositionRowIndex,currentPositionButNotRowAware);
		text.setSelection(currentPosition, currentPosition);
		
		setDirty(true);

	}
	
	
	/**
	 * determine which is closer, a protected character or the end of the line
	 * based on the caretPositon 
     * 
	 * @return the cursor position (not the offset) that contains the last place to type/paste 
	 */
	protected int getLastCursorPositionForDeletion(String source, int caretPosition, ArrayList<Integer> characterPositionsThatCantBeModified) {
		int terminatingCharacterPosition = (int) maxLengthP;
		
		int currentPositionThatIncludesOffset = caretPosition;
		for(Integer x : characterPositionsThatCantBeModified) {
			if(x >= currentPositionThatIncludesOffset) {
				// found the closet terminating character
				terminatingCharacterPosition = x;
				break;
			}
		}
		return terminatingCharacterPosition;
	}
	
	protected int getTruePositionBasedOnRowIndexAndPositionOffset(int rowIndex, int caretOffset) {
		if(rowIndex == 0) {
			return caretOffset;
		} else if(rowIndex == 1) {
			return textFromWidgetSplitIntoRows[0].length() + LINE_SEPARATOR.length() + caretOffset;
		} else if(rowIndex == 2) {
			return 2* textFromWidgetSplitIntoRows[0].length() + 2* LINE_SEPARATOR.length() + caretOffset;
		}
		return -1;
	}

	public void setMaxLengthP(int maxLengthP) {
		this.maxLengthP = maxLengthP;			
	}
	
	public void setProtectedValues(ArrayList<Integer> protectedValues) {
		this.protectedValues = protectedValues;
	}
	
	/**
	 * 20130905 - carel06 - 2735
	 * when trying to return a character from position not available, return null
	 * 
	 * If trying to extract a character from a text control
	 * that doesn't have that character availabe, because the text control length
	 * is too short, just replace it with a space.
	 * 
	 * The use case for this is that I deleted the cust-name field so that it didn't have any characters.
	 * Mainframe response is just an empty CharData.  Even though max-length-p is 15, it returns null.
	 * 
	 * @return a character from the text control, if it cannot access return a space.
	 */
	private Character getCharacterFromTextControl(String value) {
		try {
			return new Character(value.charAt(textCaretPosition));
		} catch (StringIndexOutOfBoundsException e) {
			return null;
		}
	}
	
	protected void isEndOfRecordField(KeyEvent keyEvent) {
		int currentCursorPositionInRecordField = textCaretPositionOffset;

		if(currentCursorPositionInRecordField >= maxLengthP) {
            // forward acting and already at the end of the record field, do nothing
			keyEvent.doit = false;
			return;
		}
	}
	
	/**
	 * Exposing the handleNonNullCharacter to child classes.
	 * The reason for this is that in Character Mode Key Column,
	 * only a digit is accepted.  I want to have a check for this in 
	 * one of the child classes.
	 * 
	 * @param keyEvent
	 */
	protected void handleNonNullCharacter(KeyEvent keyEvent) {	
		Text text = (Text) keyEvent.getSource();
		
		if(insertToggleFlag || textSelection.x != textSelection.y) {
			handleInsertMode(keyEvent, text, text.getCaretPosition());
		} else {
			handleOvertypeMode(keyEvent, text, text.getCaretPosition());
		}

		keyEvent.doit = false;
	}
	
	private void handleInsertMode(KeyEvent keyEvent, Text text, int caretPosition) {
		String stringContentOfTheTextWidget = text.getText();

		String[] row = text.getText().split(LINE_SEPARATOR);

		if(textCaretPositionRowIndex > 2) {
			return;
		}

		boolean insertableFlag = false;
		insertableFlag = isInsertable(row[textCaretPositionRowIndex], caretPosition, 1, protectedValues);

		if(insertableFlag) {
            String valueOfText = new String(row[textCaretPositionRowIndex]);

            int terminatingCharacterPosition = -1;
            if(!stringContentOfTheTextWidget.isEmpty()) {
            	try {
            		terminatingCharacterPosition = getLastCursorPositionForInsertion(valueOfText, caretPosition);                        
            		valueOfText = deleteCharacterWhileEditing(valueOfText, terminatingCharacterPosition);
            	} catch (StringIndexOutOfBoundsException e) {
            	}
            }
      
            int numberOfCharactersSelection = Math.abs(textSelection.x - textSelection.y);
            if(numberOfCharactersSelection > 0 && textSelection.y == textCaretPosition && textCaretPositionRowIndex != 0)  {
            	// if there is no textSelection
            	textCaretPositionOffset = textCaretPositionOffset - numberOfCharactersSelection;
            } 
            int offest = textCaretPositionRowIndex * (textFromWidgetSplitIntoRows[textCaretPositionRowIndex] + LINE_SEPARATOR).length();
            
            valueOfText = valueOfText.substring(0, textSelection.x - offest) + keyEvent.character + valueOfText.substring(textSelection.x - offest, valueOfText.length());
            row[textCaretPositionRowIndex] = valueOfText.toString();
            
            
            if(row.length > 1) {
            	text.setText(row[0] + LINE_SEPARATOR + row[1] + LINE_SEPARATOR + row[2]);                                        
            } else {
            	text.setText(row[0]);
            }

            int correctCaretPositionEvenIfThereIsSelection = caretPosition;
            if(numberOfCharactersSelection > 0) {
            	if(textSelection.x == textCaretPosition) {
            		correctCaretPositionEvenIfThereIsSelection = textCaretPosition;
            	} else {
            		correctCaretPositionEvenIfThereIsSelection = textCaretPosition - numberOfCharactersSelection;
            	}
            } 
            moveToNextCharacter(text, correctCaretPositionEvenIfThereIsSelection, protectedValues);
            
            // the text value has been change, make the editor dirty
            setDirty(true);
		} else {
		}
	}
	
	/**
	 * This method handles the overtype mode.
	 * once the text has been replaced with the new StringBuffer value
	 * we need to position the cursor in the right position.
	 * this does replacement of characters
	 * 
	 * @param keyEvent
	 * @param text
	 * @param caretPosition
	 */
	private void handleOvertypeMode(KeyEvent keyEvent, Text text, int caretPosition) {
		// this replaces the character in the current position with the one 
		// newly typed.  also need to update the cursor position
		String valueOfText = deleteCharAtPositionAndInsertCharAtPosition(keyEvent, text, caretPosition);
		text.setText(valueOfText.toString());	
		moveToNextCharacter(text, caretPosition, protectedValues);
		setDirty(true);
	}
	
	/**
	 * Insert of using a stringBuffer, we are using the String class instead.
	 * StringBuffer adds extra characters at the end and we don't want to use that.
	 * 
	 * @param keyEvent
	 * @param text
	 * @param caretPosition
	 * @return
	 */
	private String deleteCharAtPositionAndInsertCharAtPosition(KeyEvent keyEvent, Text text, int caretPosition) {	
		textFromWidgetSplitIntoRows[textCaretPositionRowIndex] = textFromWidgetSplitIntoRows[textCaretPositionRowIndex].substring(0, textCaretPositionOffset) + textFromWidgetSplitIntoRows[textCaretPositionRowIndex].substring(textCaretPositionOffset + 1, textFromWidgetSplitIntoRows[textCaretPositionRowIndex].length());
		textFromWidgetSplitIntoRows[textCaretPositionRowIndex] = textFromWidgetSplitIntoRows[textCaretPositionRowIndex].substring(0, textCaretPositionOffset) + keyEvent.character + textFromWidgetSplitIntoRows[textCaretPositionRowIndex].substring(textCaretPositionOffset, textFromWidgetSplitIntoRows[textCaretPositionRowIndex].length());
		
		if(textFromWidgetSplitIntoRows.length > 1) {
			return textFromWidgetSplitIntoRows[0] + LINE_SEPARATOR + textFromWidgetSplitIntoRows[1] + LINE_SEPARATOR + textFromWidgetSplitIntoRows[2];
		} else {
			return textFromWidgetSplitIntoRows[0];			
		}
	}
	
	/**
	 * carel06 / Defect 2954
	 * Had to create a sepate method for this since
	 * it is being handled differently in CharMode vs. SingleRecordFormatMode.
	 * 
	 * @param valueOfText
	 * @param terminatingCharacterPosition
	 * @return
	 */
	protected String deleteCharacterWhileEditing(String valueOfText, int terminatingCharacterPosition) {
		String firstSubString = valueOfText.substring(0   , terminatingCharacterPosition - 1);
		String secondSubString = valueOfText.substring(terminatingCharacterPosition, valueOfText.length());
		valueOfText = firstSubString + secondSubString;
		return valueOfText;
	}
	
	/**
     * 2014/10/21 carel06
     * Added the characterPositionsThatCantBeModified variable since it is a 
     * different behavior when in hexData vs. charData.
     * 
	 * To be able to insert, we must have some spaces at the end of the record field.
	 * We calculate the number of available spaces.
	 * We figure out how many spaces we can insert to based on current caret position.
	 * And we do a quick compare if we have enough room.
	 * 
	 * @param text - string to be pasted upon - destination 
	 * @param caretPosition - The caret position provided is 0-based
	 * @param numberOfCharactersToInsert
     * @param characterPositionsThatCantBeModified the positions that are protected
	 * @return 
	 */
	protected boolean isInsertable(String text, int caretPosition, int numberOfCharactersToInsert, ArrayList<Integer> characterPositionsThatCantBeModified) {
		String textSubstringThatAccountsForCaretPositionAndProtectedValues;
		int positionThatContainsProtectedValue = -1;
		//if selection is protected area don't paste
		
		if(textSelection.x == textSelection.y) {
			// there is no selection
			if(characterPositionsThatCantBeModified.contains(textCaretPosition)) {
				return false;
			}					
		} else {

			if(characterPositionsThatCantBeModified.contains(((int) textSelection.x)) || characterPositionsThatCantBeModified.contains(((int) textSelection.y)-1)) {
				return false;
			}			
		}
		
		for(Integer x: characterPositionsThatCantBeModified) {

			int currentPositionModel = textCaretPosition;
			if(x > currentPositionModel) {
				positionThatContainsProtectedValue = x.intValue();
				break;
			}
		}

		if(characterPositionsThatCantBeModified.size() == 0 || positionThatContainsProtectedValue == -1) {
			if(textCaretPositionRowIndex>0){
			textSubstringThatAccountsForCaretPositionAndProtectedValues = text; //.substring(textCaretPositionOffset);
			}else{
				textSubstringThatAccountsForCaretPositionAndProtectedValues = text.substring(caretPosition);	
			}

		} else {
			textSubstringThatAccountsForCaretPositionAndProtectedValues = text.substring(textSelection.x, positionThatContainsProtectedValue);
		}
		
		int numberOfSpacesBeforeTerminatingPosition = getNumberOfTrailingSpaces(textSubstringThatAccountsForCaretPositionAndProtectedValues); 
		numberOfSpacesBeforeTerminatingPosition = numberOfSpacesBeforeTerminatingPosition +  textFromKeyPressed.getSelectionCount();
		if(numberOfSpacesBeforeTerminatingPosition >= numberOfCharactersToInsert) {
			return true;
		} else {
			return false;
		}
	}
	
	private int getNumberOfTrailingSpaces(String str) {
		int i;
		int numberOfSpaces = 0;
		for ( i=str.length()-1;i>=0;i--){  
			char c = str.charAt(i);  
			if (c != EMPTY_SPACE_TO_INSERT) {  
				break;  
			} else {
				numberOfSpaces++;
			}
		}  
		return numberOfSpaces;
	}
	
	protected int getLastCursorPositionForInsertion(String source, int caretPosition) {
		return getLastCursorPositionForDeletion(source, caretPosition, protectedValues);
	}
	
	public void setDirty(boolean isDirty){
			this.isDirty = isDirty;
	}

	public boolean getDirty(){
		return isDirty;
	}
}
