/**
 * Copyright (c) 2017 CA, All rights Reserved.

This software and all information contained therein is confidential and 
proprietary and shall not be duplicated, used, disclosed or disseminated in any 
way except as authorized by the applicable license agreement, without the 
express written permission of CA. All authorized reproductions must be marked 
with this language.  

EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT PERMITTED 
BY APPLICABLE LAW, CA PROVIDES THIS SOFTWARE WITHOUT WARRANTY OF ANY KIND, 
INCLUDING WITHOUT LIMITATION, ANY IMPLIED WARRANTIES OF MERCHANTABILITY OR 
FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT WILL CA BE LIABLE TO THE END USER 
OR ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE OF 
THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, LOST PROFITS, BUSINESS 
INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF CA IS EXPRESSLY ADVISED OF SUCH 
LOSS OR DAMAGE.

 ***********************************************************************/
package com.ca.fmp.ims.view.dialogs;

import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;

import com.ca.fmp.ims.model.generated.IMSENVInfoType;
import com.ca.testingtools.ui.TextUtil;
import com.ca.fmp.ims.common.VPATSortDialog;  //grake02 25/OCT/2016

/**
 * This dialog opens up when trying to select an IMS Environment.
 * 
 * @author carel06
 *
 */
public class WildCardMemberDialog extends TitleAreaDialog {

    
	private Logger log = Logger.getLogger(WildCardMemberDialog.class.getName());
    private static String CONTEXT_HELP_ID = "com.ca.fmp.ims.help.imsEnv"; //$NON-NLS-1$
    private static String TITLE = "Select IMS Environment";
    private static String MESSAGE_PREFIX = "Select member from ";
    private ArrayList<IMSENVInfoType> environmentMemberInfoList;
	private TableViewer tableViewer;
	private String dsnName;
	private Combo imsEnvironmentCombo;
	private IMSENVInfoTypeComparator comparator;
	
	private static Listener sortListenerVPAT;  //grake02 24/OCT/2016 - listen for keys to invoke column sort dialog

	/**
	 * @param parentShell
	 * @param dsnName
	 * @param environmentMemberInfoList - list of IMSEnvInfoType to display
	 * @param imsEnvironmentCombo - need to provide the combo so that after pressing the ok button, the combo widget is updated with the correct ims environment
	 */
	public WildCardMemberDialog(Shell parentShell, String dsnName, ArrayList<IMSENVInfoType> environmentMemberInfoList, Combo imsEnvironmentCombo) {
		super(parentShell);
		this.environmentMemberInfoList = environmentMemberInfoList;
		this.dsnName = dsnName;
        this.imsEnvironmentCombo = imsEnvironmentCombo;
		
		// we need to set the style for this shell so that 
		// we can have the help tray dialog open up.
        int style = SWT.CLOSE | SWT.TITLE | SWT.MIN | SWT.RESIZE;
        setShellStyle(style | SWT.APPLICATION_MODAL);       
        
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		this.getShell().setText("Select IMS Environment");
		
		Composite area = (Composite) super.createDialogArea(parent);
		Composite container = new Composite(area, SWT.NONE);
		container.setLayoutData(new GridData(GridData.FILL_BOTH));
		GridLayout layout = new GridLayout(1, false);
		container.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		container.setLayout(layout);

		setTitle(TITLE);
		setMessage(MESSAGE_PREFIX + "'" + dsnName + "'");

		createViewer(container);

		// need to display the amount of members being displayed
		Text memberCount = new Text(container, SWT.READ_ONLY);
		memberCount.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, true, false, 1, 1));
        TextUtil.setWidth(memberCount, 14);
        memberCount.setText(environmentMemberInfoList.size() + " members returned.");
        
		// set the sorter for the table
		comparator = new IMSENVInfoTypeComparator();
		tableViewer.setComparator(comparator);

		setHelpAvailable(true);
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, CONTEXT_HELP_ID);

		// after opening the dialog,
		// the first row is selected immediately.
		// Ken found this bug since it was a different behavior from
		// the select DBD dialog
		tableViewer.getTable().setFocus();
		tableViewer.getTable().setSelection(0);
		
        //grake02 24/OCT/2016 - invoke sort dialog so that application is 508 compliant
		setupVPATColumnSorter();

		return area;
	}

	private void createViewer(Composite parent) {
		tableViewer = new TableViewer(parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		createColumns(parent, tableViewer);
		final Table table = tableViewer.getTable();		                
		table.setLinesVisible(true);
		table.setHeaderVisible(true);

		tableViewer.setContentProvider(new ArrayContentProvider());
		// get the content for the tableViewer, setInput will call getElements in the
		// contentProvider
		
		ArrayList<IMSENVInfoType> d = new ArrayList<IMSENVInfoType>();
		for(IMSENVInfoType x: environmentMemberInfoList) {
			d.add(x);
		}	
		tableViewer.setInput(d);
		
		// make the selection available to other views
		// getSite().setSelectionProvider(tableViewer);
		// set the sorter for the table

		// define layout for the tableViewer
		GridData gridData = new GridData();
		gridData.verticalAlignment = GridData.FILL;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.horizontalSpan = 2;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.heightHint = 300;
		tableViewer.getControl().setLayoutData(gridData);
		
 
		
	}
	
	// create the columns for the table
	private void createColumns(final Composite parent, final TableViewer viewer) {
		String[] titles = { "Name", "Type", "Description"};
		int[] bounds = { 100, 100, 300, 100 };

		// first column is for the dataset name
		TableViewerColumn col = createTableViewerColumn(titles[0], bounds[0], 0);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				IMSENVInfoType p = (IMSENVInfoType) element;
		        return p.getName();				
			}
		});
		
		// second column is DLI/BMP 
	    col = createTableViewerColumn(titles[1], bounds[1], 1);
	    col.setLabelProvider(new ColumnLabelProvider() {
	      @Override
	      public String getText(Object element) {
	    	  IMSENVInfoType p = (IMSENVInfoType) element;
	        return p.getType().name();
	      }
	    });
	    
	    // third column is the description 
	    col = createTableViewerColumn(titles[2], bounds[2], 2);
	    col.setLabelProvider(new ColumnLabelProvider() {
	    	@Override
	    	public String getText(Object element) {
	    		IMSENVInfoType p = (IMSENVInfoType) element;
	    		return p.getDescription();
	    	}
	    });
	}
	
	private TableViewerColumn createTableViewerColumn(String title, int bound, final int colNumber) {
		final TableViewerColumn viewerColumn = new TableViewerColumn(tableViewer, SWT.NONE);
		final TableColumn column = viewerColumn.getColumn();
		column.setText(title);
		column.setWidth(bound);
		column.setResizable(true);
		column.addSelectionListener(getSelectionAdapter(column, colNumber));
		return viewerColumn;
	}

	private SelectionAdapter getSelectionAdapter(final TableColumn column, final int index) {
		SelectionAdapter selectionAdapter = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				comparator.setColumn(index);
				int dir = comparator.getDirection();
				tableViewer.getTable().setSortDirection(dir);
				tableViewer.getTable().setSortColumn(column);
				tableViewer.refresh();
			}
		};
		return selectionAdapter;
	}
	/**
	 * Set the combo box with the selected ims environment
	 */
	@Override
	protected void okPressed() {
		IMSENVInfoType environmentMemberInfoType = (IMSENVInfoType) tableViewer.getElementAt(tableViewer.getTable().getSelectionIndex());
		imsEnvironmentCombo.setText(environmentMemberInfoType.getName());
		super.okPressed();
	}
	
	//grake02 24/OCT/2016
	@Override
	public boolean close()
	{
	    getShell().getDisplay().removeFilter(SWT.KeyDown, sortListenerVPAT);
	    return super.close();
	}	
	
	//grake02 26/OCT/2016 - create filter filter for 508 compliant problem.
	private void setupVPATColumnSorter()
	{
        //grake02 26/OCT/2016 - invoke sort dialog so that application is 508 compliant
		sortListenerVPAT = new Listener() 
        {
        	public void handleEvent(Event event) 
        	{
	            if (event.keyCode == SWT.F12)
	            {
	            	VPATSortDialog VPATdialog = new VPATSortDialog(getShell(), tableViewer);
	            	VPATdialog.create();
	            	if (VPATdialog.open() == Window.OK)
	            	{
	            		comparator.setColumn(tableViewer.getTable().indexOf((tableViewer.getTable().getSortColumn())));  //grake02 3/NOV/2016
	            		tableViewer.refresh();
	            	}
	            }
	        }
	    };
	    
	    getShell().getDisplay().addFilter(SWT.KeyDown, sortListenerVPAT);      				
	}	
}
